using System;
using System.Data;
using System.IO;

namespace gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages
{

		#region Header

		//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		//<Developers>
		//	<Developer>Brian Tomlin</Developer>
		//</Developers>
		//<SiteName>Hines OIFO</SiteName>
		//<CreationDate>10/19/2004</CreationDate>
		//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		//<summary>The HL7OmgMessage type represents the HL7 v2.4 Response message(ORG) to a General Clinical Order Message(OMG). It implements the HL7ProtocolMessage type and is not meant to be inherited.</summary>

		#endregion

	/// <summary>
	/// Class HL7OrgMessage
	/// </summary>
	public sealed class HL7OrgMessage : HL7ProtocolMessage
	{
		/// <summary>
		/// Response message(ORG) to a General Clinical Order Message(OMG)
		/// </summary>
		private const string MESSAGE_TYPE = "ORG^O20";

		private const bool RESPONSE_REQUIRED_INDICATOR = false;

		private string _errorText;
		private string _ackCode;
		private string _msh;
		private string _msa;
		private string _pid;
		private string _orc;
		private string _obr;

		/// <summary>
		/// Empty static constructor added to get rid of "beforefieldinit" attribute generated by compiler.
		/// </summary>
		static HL7OrgMessage() {}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="378"> 
		///		<ExpectedInput>Input string</ExpectedInput>
		///		<ExpectedOutput>HL7OrgMessage object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="381"> 
		///		<ExpectedInput>NULL message input string</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor to load HL7 Message.
		/// </summary>
		/// <param name="message"></param>
		public HL7OrgMessage( string message )
			: base( message )
		{
			if( message == null )
			throw( new ArgumentNullException( "message" ) );

			_message = message;

			_messageControlID = HL7Utility.GetMessageControlID( message );
			LoadDataFromHL7Message();
		}

		private void LoadDataFromHL7Message()
		{
			_msh = HL7Utility.ParseGetRequiredMessageSegment(this.Message, SegmentTypeNames.MSH);
			_msa = HL7Utility.ParseGetRequiredMessageSegment(this.Message, SegmentTypeNames.MSA);
			_pid = HL7Utility.ParseGetOptionalMessageSegment(this.Message, SegmentTypeNames.PID);
			_orc = HL7Utility.ParseGetOptionalMessageSegment(this.Message, SegmentTypeNames.ORC);
			_obr = HL7Utility.ParseGetOptionalMessageSegment(this.Message, SegmentTypeNames.OBR);

			_ackCode = HL7Utility.ParseGetAckCode( this.Message );
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="383"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>Message string used as imput message in constructor.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="386"> 
		///		<ExpectedInput>No Failure conditions apply.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overridden method to get the full hl7 message.
		/// </summary>
		/// <returns></returns>
		public override string GetMessage()
		{
			return Message;
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="388"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="391"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method always returns true for this type of message.
		/// </summary>
		/// <returns>boolean</returns>
		public override bool IsResponseRequired()
		{
			return RESPONSE_REQUIRED_INDICATOR;
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="392"> 
		///		<ExpectedInput>Random number in Message Control ID field in HL7 OMG message.</ExpectedInput>
		///		<ExpectedOutput>Same number as Message Control ID in input message.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="395"> 
		///		<ExpectedInput>No failure conditions</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the Message Control ID associated with this message
		/// </summary>
		/// <returns>Message Control ID in MSH segment</returns>
		public override string GetMessageControlID()
		{
			return MessageControlID;
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="396"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="397"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overriden method to get the Message Type.
		/// </summary>		
		public override string GetMessageType()
		{
			return MessageType;
		}

		/// <summary>
		/// WriteHL7MessageBody
		/// </summary>
		/// <param name="writer"></param>
		protected override void WriteHL7MessageBody( StreamWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.Write( this.Message );
		}

		
		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="398"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="399"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Message Type.
		/// </summary>
		public string MessageType
		{
			get
			{
				return MESSAGE_TYPE;
			}

		}


		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="401"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="402"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Full HL7 message
		/// </summary>
		public string Message
		{
			get
			{
				return _message;
			}
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="403"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="404"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Message Control ID
		/// </summary>
		public string MessageControlID
		{
			get
			{
				return _messageControlID;
			}
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="405"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="408"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Error Text
		/// </summary>
		public string ErrorText
		{
			get
			{
				return _errorText;
			}
			set
			{
				_errorText = value;
			}
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="409"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="412"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Order control code
		/// </summary>
		public string AcknowledgementCode
		{
			get
			{
				return _ackCode;
			}
			set
			{
				_ackCode = value;
			}
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="413"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="416"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MSH Segment
		/// </summary>
		public string MSH
		{
			get
			{
				return _msh;
			}
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="417"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="420"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MSA Segment
		/// </summary>
		public string MSA
		{
			get
			{
				return _msa;
			}
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="422"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="425"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PID Segment
		/// </summary>
		public string PID
		{
			get
			{
				return _pid;
			}
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="431"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="432"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ORC Segment
		/// </summary>
		public string ORC
		{
			get
			{
				return _orc;
			}
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="437"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="438"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OBR Segment
		/// </summary>
		public string OBR
		{
			get
			{
				return _obr;
			}
		}
	}
}
